/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include "common.h"
#include "version.h"
#include "context.h"
#include <sstream>

#define TEMP_FILE_MASK "%%%%-%%%%-%%%%-%%%%.tmp"

#define APP "app"
#define CONTEXT "context"
#define V "version"

bool compare_no_case (const std::string& a, const std::string& b)
{   if (a.length () != b.length ()) return false;
    return ::std::equal (b.begin (), b.end (), a.begin (), cnc_test); }

bool is_one_of (const ::std::string& s, const vstr_t& v)
{   for (::std::string i : v)
        if (compare_no_case (s, i)) return true;
    return false; }

::std::string read_text_file (const::std::string& name)
{   using namespace boost::filesystem;
    path p (name);
    if (exists (p) && is_regular_file (p))
    {  ifstream f (name);
        if (! f.bad ())
        {   ::std::stringstream res;
            res << f.rdbuf ();
            f.close ();
            return res.str (); } }
    return ::std::string (); }

bool write_text_file (const ::std::string& name, const ::std::string& content)
{   using namespace boost::filesystem;
    path n (name);
    path p (n);
    p += ".tmp";
    try
    {   ofstream f (p);
        if (f.bad ())
        {   if (context.tell (e_severe)) ::std::cerr << "Cannot open temporary file " << p << "\n";
            return false; }
        try
        {   f << content; }
        catch (...)
        {   if (context.tell (e_severe)) ::std::cerr << "Cannot write to temporary file " << p << "\n";
            return false; }
        f.close ();
        if (::boost::filesystem::exists (n))
            if (! ::boost::filesystem::remove (n))
            {   if (context.tell (e_severe)) ::std::cerr << "Cannot delete existing file " << n << "\n";
                return false; }
        ::boost::filesystem::rename (p, n);
        return true; }
    catch (...) { }
    try
    {   if (::boost::filesystem::exists (p))
            ::boost::filesystem::remove (p); }
    catch (...) { }
    if (context.tell (e_error)) ::std::cerr << "Cannot update " << n << "\n";
    return false; }

::std::string trim_the_lot_off (const ::std::string& s)
{   return ::boost::trim_copy (s); }

vstr_t split_by_charset (const ::std::string& s, const char* charset)
{   vstr_t v;
    ::boost::algorithm::split (v, s, ::boost::algorithm::is_any_of (charset), ::boost::algorithm::token_compress_on);
    return v; }

bool remove_tail (::std::string& s, ::std::string& tail, const char ch)
{   const ::std::string::size_type pos = s.find (ch);
    tail.clear ();
    if (pos == ::std::string::npos) return false;
    if (pos < s.length () - 1) tail = trim_the_lot_off (s.substr (pos + 1));
    if (pos == 0) s.clear ();
    else s = trim_the_lot_off (s.substr (0, pos));
    return true; }

::std::string remove_tail (::std::string& s, const char ch)
{   ::std::string tail;
    const ::std::string::size_type pos = s.find (ch);
    if (pos != ::std::string::npos)
    {   if (pos < s.length () - 1) tail = trim_the_lot_off (s.substr (pos + 1));
        if (pos == 0) s.clear ();
        else s = trim_the_lot_off (s.substr (0, pos)); }
    return tail; }

bool remove_head (::std::string& s, ::std::string& head, const char ch)
{   const ::std::string::size_type pos = s.find (ch);
    head.clear ();
    if (pos == ::std::string::npos) return false;
    if (pos > 0) head = trim_the_lot_off (s.substr (0, pos));
    if (pos < s.length () - 1) s = trim_the_lot_off (s.substr (pos + 1));
    else s.clear ();
    return true; }

::std::string remove_head (::std::string& s, const char ch)
{   ::std::string head;
    const ::std::string::size_type pos = s.find (ch);
    if (pos != ::std::string::npos)
    {   if (pos > 0) head = trim_the_lot_off (s.substr (0, pos));
        if (pos < s.length () - 1) trim_the_lot_off (s = s.substr (pos + 1));
        else s.clear (); }
    return head; }

bool separate (const ::std::string& s, ::std::string& head, ::std::string& tail, const char ch)
{   tail.clear ();
    const ::std::string::size_type pos = s.find (ch);
    if (pos == ::std::string::npos) { head = s; return false; }
    if (pos > 0) head = trim_the_lot_off (s.substr (0, pos)); else head.clear ();
    if (pos < s.length () - 1) tail = trim_the_lot_off (s.substr (pos + 1));
    return true; }


::boost::filesystem::path get_tmp_filename ()
{   ::boost::system::error_code ec;
    ::boost::filesystem::path model (::boost::filesystem::temp_directory_path (ec));
    if (ec.failed ()) return ::boost::filesystem::path ();
    model /= TEMP_FILE_MASK;
    return ::boost::filesystem::unique_path (model); }

bool read_header (::boost::property_tree::ptree& json, const ::std::string& expected, ::std::string& version, const ::std::string& filename)
{   ::std::string prog = read_field < ::std::string > (json, APP);
    version = read_field < ::std::string > (json, V);
    ::std::string con = read_field < ::std::string > (json, CONTEXT);
    if ((prog != PROG) || (version.substr (0, 3) != "0.0"))
    {   if (context.tell (e_error)) ::std::cerr << filename << " is not an " PROG " file, or this copy of " PROG " (v" VERSION_STRING ") is too old to read it\n";
        return false; }
    if (! expected.empty ())
        if (con != expected)
        {   if (context.tell (e_error)) ::std::cerr << filename << " is not an " PROG " " << expected << " file\n";
            return false; }
    return true; }

void write_header (::boost::property_tree::ptree& json, const char* k)
{   write_field < ::std::string > (json, APP, PROG);
    write_field < ::std::string > (json, V, VERSION_STRING);
    write_field < ::std::string > (json, CONTEXT, k); }

bool replace_file (::boost::property_tree::ptree& json, ::boost::filesystem::path& filename)
{   ::boost::filesystem::path tmp (filename), old (filename);
    tmp += ".tmp";
    old += ".old";
    if (! ::boost::filesystem::exists (filename))
    {   try
        {   ::boost::property_tree::write_json (filename.string (), json); }
        catch (...)
        {   ::boost::filesystem::remove (filename);
            if (context.tell (e_severe)) ::std::cerr << "Cannot write " << filename << "\n";
            return false; } }
    else
    {   try
        {   ::boost::property_tree::write_json (tmp.string (), json);
            ::boost::filesystem::rename (filename, old); }
        catch (...)
        {   ::boost::filesystem::remove (tmp);
            if (context.tell (e_severe)) ::std::cerr << "Cannot write " << tmp << "\n";
            return false; }
        try
        {   ::boost::filesystem::rename (tmp, filename); }
        catch (...)
        {   ::boost::filesystem::rename (old, filename);
            ::boost::filesystem::remove (tmp);
            if (context.tell (e_severe)) ::std::cerr << "Cannot replace " << filename << " with " << tmp << "\n";
            return false; }
       try
        {   ::boost::filesystem::remove (old); }
        catch (...) { } }
    return true; }
