/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once

#include "element_attribute.h"
#include <tuple>

class html_element_base
{
protected:
    size_t tag_ = MyHTML_TAG__UNDEF;
    myhtml_element el_;
    ::std::string unrecognised_;
    bool parse (myhtml_tree_attr_t* a);
    virtual bool subparse (myhtml_attribute& ) { return false; }
public:
    html_element_base () = default;
    explicit html_element_base (const size_t tag) : tag_ (tag) { }
    explicit html_element_base (const myhtml_element& e) : el_ (e) { }
    explicit html_element_base (myhtml_tree_node_t* e) : el_ (e) { }
    virtual bool has (const e_attribute ) const { return false; }
    virtual ::std::string get_value (const e_attribute ) const { return ::std::string (); }
    virtual bool set_value (const e_attribute , const ::std::string* ) const { return false; }
    virtual size_t whoami () const { return tag_; }
    virtual void set_tag (const size_t x) { tag_ = x; }
    virtual void swap (html_element_base& el) noexcept
    {   ::std::swap (tag_, el.tag_);
        unrecognised_.swap (el.unrecognised_);
	    el_.swap (el.el_); }
    virtual void reset ()
    {   html_element_base el;
        swap (el); }
    ::std::string content (bool text, bool anything);
    virtual bool empty () const { return true; }
    virtual bool invalid () const { return false; }
    virtual bool invalid (const e_attribute ) const { return false; }
    virtual bool valid () const { return false; }
    virtual bool valid (const e_attribute ) const { return false; }
    virtual void verify (sstr_t* ) { }
    virtual bool invalid_id (sstr_t* ) { return false; }
    virtual bool has_url () const { return false; }
    virtual bool verify_url (const directory& ) { return false; }
    virtual ::std::string diagnose (const int ) const { return ::std::string (); }
    virtual ::std::string report (const int ) const { return ::std::string (); } };


template < class ATTRIBUTE, class... ATTRIBUTES > struct attribute_test : attribute_test < ATTRIBUTES... >
{   static bool has (const e_attribute a)
    {   if (a == ATTRIBUTE::whoami ()) return true;
        return attribute_test < ATTRIBUTES... > :: has (a); }
    template < class T > static ::std::string get_value (const T& t, const e_attribute a)
    {   if (a == ATTRIBUTE::whoami ()) return ::std::get < ATTRIBUTE > (t).get_value ();
        return attribute_test < ATTRIBUTES... > :: get_value (t, a); }
    template < class T > static bool set_value (T& t, const e_attribute a, const ::std::string& s)
    {   if (a != ATTRIBUTE::whoami ()) return attribute_test < ATTRIBUTES... > :: set_value (t, a, s);
        ::std::get < ATTRIBUTE > (t).set_value (s);
        return true; } };

template < class ATTRIBUTE > struct attribute_test < ATTRIBUTE >
{   static bool has (const e_attribute a)
    {   return (a == ATTRIBUTE::whoami ()); }
    template < class T > static ::std::string get_value (const T& t, const e_attribute a)
    {   if (a == ATTRIBUTE::whoami ()) return ::std::get < ATTRIBUTE > (t).get_value ();
        return ::std::string ("(NAV)"); }
    template < class T > static bool set_value (T& t, const e_attribute a, const ::std::string& s)
    {   if (a != ATTRIBUTE::whoami ()) return false;
        ::std::get < ATTRIBUTE > (t).set_value (s);
        return true; } };


template < size_t TAG, class... ATTRIBUTES > class html_element : public html_element_base
{   typedef ::std::tuple < ATTRIBUTES... > attributes_t;
    attributes_t a_;
protected:
    virtual bool subparse (myhtml_attribute& html);
public:
    html_element () : html_element_base (TAG) { }
    html_element (const myhtml_element& node);
    html_element (myhtml_tree_node_t* node);
    static int tag () { return TAG; }
    virtual bool has (const e_attribute a) const
    {   return attribute_test < ATTRIBUTES... > :: has (a); }
    template < class ATTRIBUTE > static bool has ()
    {   return attribute_test < ATTRIBUTES... > :: has (ATTRIBUTE::whoami ()); }
    template < class ATTRIBUTE > void get (ATTRIBUTE** p)
    {   assert (p != nullptr);
        *p = & (::std::get < ATTRIBUTE > (a_)); }
    template < class ATTRIBUTE > void get (const ATTRIBUTE** p) const
    {   assert (p != nullptr);
        *p = & (::std::get < ATTRIBUTE > (a_)); }
    template < class ATTRIBUTE > ::std::string get_value () const
    {   return ::std::get < ATTRIBUTE > (a_).get_value (); }
    template < class ATTRIBUTE > void set_value (const ::std::string& s)
    {   ::std::get < ATTRIBUTE > (a_).set_value (s); }
    virtual ::std::string get_value (const e_attribute a) const
    {   return attribute_test < ATTRIBUTES... > :: template get_value < attributes_t > (a_, a); }
    virtual bool set_value (const e_attribute a, const ::std::string& s)
    {   return attribute_test < ATTRIBUTES... > :: template set_value < attributes_t > (a_, a, s); }
    virtual void swap (html_element& el) noexcept
    {   html_element_base::swap (el);
        a_.swap (el.a_); }
    virtual bool empty () const;
    virtual bool invalid () const;
    virtual bool invalid (const e_attribute a) const;
    virtual bool valid () const { return (! invalid ()); }
    virtual bool valid (const e_attribute a) const { return has (a) && ! invalid (a); }
    virtual void verify (sstr_t* ssi);
    virtual bool invalid_id (sstr_t* ids);
    virtual bool has_url () const;
    virtual bool verify_url (const directory& d);
    virtual ::std::string diagnose (const int n) const;
    virtual ::std::string report (const int n) const; };


template < size_t TAG, class... ATTRIBUTES > html_element < TAG, ATTRIBUTES... > :: html_element (const myhtml_element& node) : html_element_base (node)
{   set_tag (TAG);
    if (el_.has_attributes ())
    {   parse (el_.attributes ());
        html_element_post_parse < html_element, TAG > :: apply (*this); } }

template < size_t TAG, class... ATTRIBUTES > html_element < TAG, ATTRIBUTES... > :: html_element (myhtml_tree_node_t* node) : html_element_base (node)
{   set_tag (TAG);
    if (el_.has_attributes ())
    {   parse (a_, el_.attributes ());
        html_element_post_parse < html_element, TAG > :: apply (*this); } }

template < size_t TAG, class... ATTRIBUTES > bool html_element < TAG, ATTRIBUTES... > :: subparse (myhtml_attribute& html)
{   bool res = false;
    if (html.unknown ())
    {   if (context.tell (e_warning))
         {   append (unrecognised_, ", ", quote (html.get_key ())); } }
    else
    {   e_attribute at = html.get_enum ();
        for_each_attribute (a_, [&](auto&& a)
        {   if (at == a.whoami ())
            {   a.parse (html);
                res = ! a.invalid (); } } ); }
    return res; }

template < size_t TAG, class... ATTRIBUTES > bool html_element < TAG, ATTRIBUTES... > :: empty () const
{   bool res = true;
    for_each_attribute (a_, [&](auto& a)
    {   res = res && a.empty (); } );
    return res; }

template < size_t TAG, class... ATTRIBUTES > bool html_element < TAG, ATTRIBUTES... > :: invalid () const
{   bool res = false;
    for_each_attribute (a_, [&](auto& a)
    {   res = res || a.invalid (); } );
    return res; }

template < size_t TAG, class... ATTRIBUTES > bool html_element < TAG, ATTRIBUTES... > :: invalid (const e_attribute at) const
{   bool res = false;
    for_each_attribute (a_, [&](auto& a)
    {   if (a.whoami () == at) res = a.invalid (); } );
    return res; }

template < size_t TAG, class... ATTRIBUTES > void html_element < TAG, ATTRIBUTES... > :: verify (sstr_t* ssi)
{   assert (ssi != nullptr);
    for_each_attribute (a_, [&](auto& a)
    {   a.verify_id (ssi); } ); }

template < size_t TAG, class... ATTRIBUTES > bool html_element < TAG, ATTRIBUTES... > :: invalid_id (sstr_t* ids)
{   assert (ids != nullptr);
    bool res = false;
    for_each_attribute (a_, [&](auto& a)
    {   if (a.invalid_id (ids)) res = true; } );
    return res; }

template < size_t TAG, class... ATTRIBUTES > bool html_element < TAG, ATTRIBUTES... > :: has_url () const
{   bool res = false;
    for_each_attribute (a_, [&](auto& a)
    {   res = res || a.is_url (); } );
    return res; }

template < size_t TAG, class... ATTRIBUTES > bool html_element < TAG, ATTRIBUTES... > :: verify_url (const directory& d)
{   bool res = true;
    for_each_attribute (a_, [&](auto& a)
    {   if (a.is_url ())
            if (! a.unknown () && ! a.empty ())
                res = res && a.verify_url (d); } );
    return res; }

template < size_t TAG, class... ATTRIBUTES > ::std::string html_element < TAG, ATTRIBUTES... > :: diagnose (const int n) const
{   ::std::string res;
    for_each_attribute (a_, [&](auto& a)
    {   if (! a.unknown ()) res += a.diagnose (); } );
    if (! res.empty ())
    {   ::std::string pre (fyi (n));
        if (! context.tell (e_comment))
        {   pre += elem::name (tag_);
            pre += ": "; }
        res = pre + res; }
    if (! unrecognised_.empty ())
    {   res += fyi (n);
        if (! context.tell (e_comment))
        {   res += elem::name (tag_);
            res += ": "; }
        res += "Unrecognised attribute(s): ";
        res += unrecognised_;
        res += "\n"; }
    return res; }

template < size_t TAG, class... ATTRIBUTES > ::std::string html_element < TAG, ATTRIBUTES... > :: report (const int n) const
{   if (! context.tell (e_debug)) return ::std::string ();
    if ((tag_ == MyHTML_TAG__UNDEF) && ! context.tell (e_splurge)) return ::std::string ();
    ::std::string res (STEPSPACES*n, ' ');
    res += elem::name (tag_);
    bool first = true;
    for_each_attribute (a_, [&](auto& a)
    {   if (! a.unknown () || context.tell (e_splurge))
        {   if (first) { res += ": "; first = false; }
            else res += ", ";
            res += a.report (); } } );
    res += "\n";
    return res; }
