/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once

#include "common.h"
#include <ctime>
#include <boost/property_tree/json_parser.hpp>

class context;

class reply
{   ::std::string file_, id_, server_, target_, content_, when_;
    ::std::ostringstream diagnosis_;
    e_activity activity_;
    ::std::string clean (const ::std::string& s);
    bool find_server (const int depth);
    bool set_server (const ::std::string& link);
public:
    reply (const ::std::string& file, const ::std::string& id, const ::std::string& target, const ::std::string& content)
        : file_ (file), id_ (id), target_ (target), content_ (clean (content)), activity_ (act_unknown)  { mark (); }
    reply (::boost::property_tree::ptree& tree, const ::std::string& container) : activity_ (act_unknown) { read (tree, container); }
    reply (const reply& r)
        :   file_ (r.file_), id_ (r.id_), server_ (r.server_), content_ (r.content_), when_ (r.when_), activity_ (r.activity_)
    {   diagnosis_ << r.diagnosis_.str (); }
    reply (reply&& ) = default;
    void swap (reply& r) noexcept
    {   file_.swap (r.file_);
        id_.swap (r.id_);
        server_.swap (r.server_);
        target_.swap (r.target_);
        content_.swap (r.content_);
        when_.swap (r.when_);
        diagnosis_.swap (r.diagnosis_);
        ::std::swap (activity_, r.activity_); }
    bool invalid () const { return file_.empty (); }
    void mark ();
    bool operator == (const reply& rhs) const;
    bool close_but_no_banana (const reply& rhs) const;
    void mark_unchanged ();
    void mark_unknown () { activity_ = act_unknown; }
    void mark_update ();
    void mark_insert ();
    void mark_delete ();
    bool is_unknown () const { return activity_ == act_unknown; }
    bool is_deleted () const { return activity_ == act_delete; }
    void read (::boost::property_tree::ptree& tree, const ::std::string& container);
    void write (::boost::property_tree::ptree& tree, const ::std::string& container);
    ::std::string report (const char* verb) const;
    ::std::string report (const size_t n) const;
    ::std::string diagnosis () const { return diagnosis_.str (); }
    bool enact (const int depth); };

typedef ::std::vector < reply > vreply_t;
const size_t no_reply = UINT_MAX;

class replies
{   vreply_t reply_;
    size_t size () const { return reply_.size (); }
    size_t find (const reply& r);
    size_t probably_match (const reply& r);
    bool read (const ::std::string filename);
    bool write ();
    bool update_records (::std::ostringstream& ss);
    bool enact (const int depth);
public:
    replies () {}
    void swap (replies& r) noexcept { reply_.swap (r.reply_); }
    void append (const ::std::string& file, const ::std::string& id, const ::std::string& target, const ::std::string& content);
    ::std::string report (const char* comment = nullptr) const;
    bool process (::std::ostringstream& ss, const int depth); };
