/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include "microformats.h"
#include "microformat_constructor.h"

bool microformats::allocated (const e_vocabulary v) const
{   return ! mf_ [v].is_empty (); }

void microformats::alloc (const e_vocabulary v)
{   assert (! allocated (v));
    mf_ [v] = microformat_pv::alloc_microformat_pv (v); }

bool microformats::has (const e_vocabulary v, const e_property p) const
{   assert (allocated (v));
    return mf_ [v].has_prop (p); }

bool microformats::has (const e_vocabulary v) const
{   return allocated (v); }

bool microformats::empty (const e_vocabulary v) const
{   if (! allocated (v)) return true;
    return mf_ [v].is_empty (); }

bool microformats::has_property (const e_vocabulary v, const e_property p) const
{   assert (allocated (v));
    return mf_ [v].has_prop (p); }

e_vocabulary microformats::plausible_vocabulary (const e_property p) const
{   for (auto mf : mf_)
        if (mf.is_allocated ())
            if (mf.get () -> has_prop (p))
                return mf.whoami ();
    return v_unknown; }

bool microformats::is_relational () const
{   for (auto mf : mf_)
        if (mf.is_allocated ())
            if (mf.get () -> is_relational ()) return true;
    return false; }

void microformats::validate_element (const size_t e) const
{   for (auto mf : mf_)
        if (mf.is_allocated ())
            mf.get () -> validate_element (e); }

bool microformats::is_declared (const e_vocabulary v) const
{   if (! allocated (v)) return false;
    for (auto mf : mf_)
        if (mf.is_allocated ())
            if (! mf.get () -> declared ()) return false;
    return true; }

void microformats::set_mf_value (const e_vocabulary v, const e_property pp, element& e)
{   assert (allocated (v));
    return mf_ [v].set_mf_value (pp, e); }

::std::string microformats::get_value (const e_vocabulary v, const e_property pp)
{   if (! allocated (v)) return ::std::string ();
    return mf_ [v].get_value (pp); }

bool microformats::empty () const
{   for (auto mf : mf_)
        if (mf.is_allocated ())
            if (! mf.get () -> empty ()) return false;
    return true; }

::std::string microformats::report (const int n) const
{   ::std::string res;
    for (auto mf : mf_)
        if (mf.is_allocated ())
            res += mf.get () -> report (n);
    return res; }

::std::string microformats::diagnose (const int n) const
{   ::std::string res;
    for (auto mf : mf_)
        if (mf.is_allocated ())
            res += mf.get () -> diagnose (n);
    return res; }

void microformats::declare (const vocab& v)
{   if (! mf_ [v].is_allocated ())
        mf_ [v] = microformat_pv::alloc_microformat_pv (v);
    if (mf_ [v].is_allocated ())
        mf_ [v].get () -> declare (); }

void microformats::reset ()
{   for (auto mf : mf_)
        mf.reset (); }
