/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include "version.h"
#include "options.h"
#include "context.h"

#define GENERAL_    "general"
#define WEBSITE_    "site"
#define LINK_       "link"
#define MF_         "microformat"
#define WMIN_       "webmention_in"
#define WMOUT_      "webmention_out"
#define SCHEMA_     "schema"

#define GENERAL     GENERAL_ "."
#define WEBSITE     WEBSITE_ "."
#define LINK        LINK_ "."
#define MF          MF_ "."
#define WMIN        WMIN_ "."
#define WMOUT       WMOUT_ "."
#define SCHEMA      SCHEMA_ "."

#define CONFIG      "config"
#define EXPORT      "export"
#define EXTENSION   "extension"
#define EXTERNAL    "external"
#define FILE        "file"
#define FORWARD     "301"
#define HELP        "help"
#define HOOK        "hook"
#define INDEX       "index"
#define LINKS       "check"
#define MICRODATA   "microdata"
#define ONCE        "once"
#define OUTPUT      "output"
#define PATH        "datapath"
#define REVOKE      "no-revoke"
#define ROOT        "root"
#define SECRET      "secret"
#define SITE        "domain"
#define VERIFY      "verify"
#define VERSION     "version"
#define VIRTUAL     "virtual"

#define CLEAR       "clear"
#define MACROSTART  "macrostart"
#define MACROEND    "macroend"
#define MENTION     "mention"
#define NOCHANGE    "nochange"
#define NOTIFY      "notify"
#define RESET       "reset"
#define STUB        "stub"
#define TEMPLATE    "template"
#define WRITE       "write"

#define TEST_HEADER "test-header"

#define EXT         "db"
#define CFG         "config"

/*
-N
picks up web mention links, digs through them, finds notification server, and sends appropriate notifications
Note: will need to keep track of mentions, so changes can be noted as additions, updates or deletes.
Mentions can be identified by their in-reply-tos, and if there are many of them to the same URL, then so be it.
Requires microformat processing

Q: Does app:
    store old pages (where?),
    talk to version control,
    require old and new page parameters,
    or have the user specify precise changes?
Q: When updating, does app:
    presume all web mentions have changed,
    or compare & look for changes?
I lean towards the latter.

-P
Make note of web mention references in files, but don't actually notify the server of changes.
Instead, prime the web mention data store so future updates can be notified

-C
Clear the web mention data store. The next time the app is run, it'll be as thought all the web mentions
are fresh and new.


-M parameters
url parameter style encoded webmention references (https://indieweb.org/Webmention-developer), expects:
source= is site containing mention to this site
target= is specific page on this site that is mentioned
both must be given

Q: Given web mentions are dynamic:
    should the target page be updated in situ
    should a set of HTML snippet be created, one per web mention, to be assembled during page compilation
    should frames or something like it be used (NO!)


sundry switches
-T insert,update,delete,unchanged
web mention HTML snippet templates

-n
tell what it would do, don't actually do it

*/


/*

also useful

1. additional HTML elements (as per data, which I've explicitly coded)
should allow attribute specification too

2. additional rel values
needs rel type and probably context info

*/

#define TYPE_HELP "Type '" PROG " -h' for help."

::std::string path_in_context (const ::std::string& file)
{   ::boost::filesystem::path res (context.path ());
    res /= file;
    return res.string (); }

void options::title (const char* addendum) const
{   ::std::cout << FULLNAME " " VERSION_STRING "\n" COPYRIGHT "\n";
     if (addendum) ::std::cout << addendum;
     ::std::cout << "\n"; }

void options::help (const ::boost::program_options::options_description& aid) const
{   ::std::cout << PROG " [switch...] file/dir..\n\n";
    ::std::cout << PROG " checks the named HTML source file(s) for broken links, dubious syntax, and microdata errors.\n";
    ::std::cout << "For a directory, it checks all appropriate files in that directory and its subdirectories.\n";
    ::std::cout << aid << "\n\n";
    ::std::cout << "If a configuration file is specified, it should be in INI file format with these optional sections:\n";
    ::std::cout << GENERAL_ ", " LINK_ ", " MF_ ", and " WEBSITE_ ".\n";
    ::std::cout << "Each section contains individual assignments using the identifiers (after the dot) noted above.\n";
    ::std::cout << "Such a file might contain:\n\n";
    ::std::cout << "[" GENERAL_ "]\n";
    ::std::cout << OUTPUT "=2\n\n";
    ::std::cout << "[" LINK_ "]\n";
    ::std::cout << LINKS "=1\n\n";
    ::std::cout << "[" WEBSITE_ "]\n";
    ::std::cout << EXTENSION "=shtml\n";
    ::std::cout << EXTENSION "=html\n";
    ::std::cout << INDEX "=index.shtml\n";
    ::std::cout << SITE "=" DEFAULT_DOMAIN "\n\n"; }


void options::process (int argc, char** argv)
{   ::boost::program_options::options_description basic ("Console Options"), primary ("Standard Options"), hidden, cmd, config, aid, line;
    ::boost::program_options::positional_options_description pos;
    pos.add (WEBSITE ROOT, 1);
    basic.add_options ()
        (HELP ",h", "output this information and exit.")
        (VERSION ",V", "display version & copyright info, then exit.")
        ;
    hidden.add_options ()
        (WMIN TEST_HEADER, ::boost::program_options::value < ::std::string > (), "use this file to test header parsing code.")
        (WMIN HOOK, ::boost::program_options::value < ::std::string > (), "process incoming " WEBMENTION ", in JSON format, in specified file.")
        (WMIN STUB, ::boost::program_options::value < ::std::string > () -> default_value ("_" PROG), "mask for file containing " WEBMENTION " HTML snippet")
        (WMIN MACROSTART, ::boost::program_options::value < ::std::string > () -> default_value ("{{"), "start of template macro")
        (WMIN MACROEND, ::boost::program_options::value < ::std::string > () -> default_value ("}}"), "start of template macro")
        (WMOUT SECRET, ::boost::program_options::value < ::std::string > (), WEBMENTION " secret.")
        (WMIN WRITE , ::boost::program_options::value < ::std::string > (), "when writing " WEBMENTION " includes, write them to this path (default: site.root).")
        (WMIN MENTION, ::boost::program_options::value < vstr_t > () -> composing (), "a " WEBMENTION ", string must be source=url,target=url. May be repeated.")
        (WMIN PATH, ::boost::program_options::value < ::std::string > () -> default_value (MENTION), "path for incoming web mention data files (note " GENERAL PATH ").") // not yet noted below
        (WMIN TEMPLATE ",t", ::boost::program_options::value < vstr_t > () -> composing (), "HTML snippets for adding mentions (new, changed, deleted, unchanged, unknown).")
        (WMOUT CLEAR ",C", "clear out all web mention data.")
        (WMOUT NOTIFY ",N", "notify appropriate servers of web mention updates.")
        (WMOUT RESET ",R", "reset web mention data.")
        (WEBSITE ROOT, ::boost::program_options::value < ::std::string > (), "website root directory (default current directory).")
        (GENERAL WEBMENTION ",w", "process webmentions.")
        ;
    line.add_options ()
        (CONFIG ",f", ::boost::program_options::value < ::std::string > () -> default_value ("." PROG "/" CFG), "load configuration from this file.")
        ;
    primary.add_options ()
        (GENERAL FILE ",c", ::boost::program_options::value < ::std::string > () -> default_value (PROG "." EXT), "file for persistent data, requires -N (note " GENERAL PATH ").")
        (GENERAL NOCHANGE ",n", "report what will do, but do not actually do it.")
        (GENERAL PATH ",p", ::boost::program_options::value < ::std::string > () -> default_value ("." PROG), "root directory for all " PROG " files.")
        (GENERAL OUTPUT ",v", ::boost::program_options::value < int > () -> default_value (static_cast <int> (default_output)), "output extra information.")

        (LINK EXTERNAL ",e", "check external links (requires curl, sets " LINK LINKS ").")
        (LINK FORWARD ",3", "report http forwarding errors, e.g. 301 and 308 (sets " LINK EXTERNAL ").")
        (LINK LINKS ",l", "check links.")
        (LINK ONCE ",o", "report each broken external link once (sets " LINK EXTERNAL ").")
        (LINK REVOKE ",r", "do not check whether https certificates have been revoked (sets " LINK EXTERNAL ").")

        (MF EXPORT, "export microformat data (only verified data if " MF VERIFY " is set).")
        (MF VERIFY ",M", "check microformats2 (see https://microformats.org/).")

        (SCHEMA MICRODATA ",m", "check microdata itemtypes.")

        (WEBSITE EXTENSION ",x", ::boost::program_options::value < vstr_t > () -> composing (), "check files with this extension (default html). May be repeated.")
        (WEBSITE INDEX ",i", ::boost::program_options::value < ::std::string > (), "index file in directories (default none).")
        (WEBSITE SITE ",s", ::boost::program_options::value < vstr_t > () -> composing (), "domain name(s) for local site (default none). May be repeated.")
        (WEBSITE VIRTUAL ",L", ::boost::program_options::value < vstr_t > () -> composing (), "define virtual directory, arg syntax directory=path. May be repeated.")
        	;
    cmd.add (basic).add (primary).add (line).add (hidden);
    config.add (primary).add (hidden);
    aid.add (basic).add (line).add (primary);
    try
    {   ::boost::program_options::store (::boost::program_options::command_line_parser (argc, argv).options (cmd).positional (pos).run (), var_); }
    catch (...)
    {   title ("\nCommand line parameter error. " TYPE_HELP "\n");
        return; }
    if (var_.count (CONFIG))
    {   ::boost::filesystem::path file (var_ [CONFIG].as < ::std::string > ());
        if (::boost::filesystem::exists (file)) try
        {   ::boost::program_options::store (::boost::program_options::parse_config_file (file.string ().c_str (), config, true), var_); }
        catch (...)
        {   ::std::cout << "Error in " << file << "\n";
            return; } }
    ::boost::program_options::notify (var_);
    if (var_.count (VERSION)) { title (); valid_ = true; return; }
    if (var_.count (HELP)) { title (); help (aid); valid_ = true; return; }
    valid_ = var_.count (WEBSITE ROOT);
    if (! valid_) title ("\n" TYPE_HELP "\n");
    else stop_ = false; }

void options::contextualise ()
{   context.clear (var_.count (WMOUT CLEAR));

    context.path (var_ [GENERAL PATH].as < ::std::string > ());
    if (! ::boost::filesystem::exists (context.path ()))
    {   ::std::cerr << context.path () << " does not exist, am creating it.";
        ::boost::filesystem::create_directories (context.path ()); }   // if throws, then exit

    context.nochange (var_.count (GENERAL NOCHANGE));
    context.persisted (path_in_context (var_ [GENERAL FILE].as < ::std::string > ()));

    context.forwarded (var_.count (LINK FORWARD));
    context.revoke (var_.count (LINK REVOKE));
    context.mf_export (var_.count (MF EXPORT));
    context.mf_verify (var_.count (MF VERIFY));
    context.microdata (var_.count (SCHEMA MICRODATA));
    context.once (var_.count (LINK ONCE));
    context.external (var_.count (LINK EXTERNAL));
    context.links (var_.count (LINK LINKS));
    context.process_webmentions (var_.count (GENERAL WEBMENTION));

    if (var_.count (GENERAL OUTPUT)) context.output (static_cast < e_output > (var_ [GENERAL OUTPUT].as < int > ()));

    if (var_.count (WEBSITE INDEX)) context.index (var_ [WEBSITE INDEX].as < ::std::string > ());
    if (var_.count (WEBSITE EXTENSION)) context.extensions ( var_ [WEBSITE EXTENSION].as < vstr_t > ());
    if (var_.count (WEBSITE ROOT)) context.root (var_ [WEBSITE ROOT].as < ::std::string > ());
    if (var_.count (WEBSITE SITE)) context.site (var_ [WEBSITE SITE].as < vstr_t > ());
    if (var_.count (WEBSITE VIRTUAL)) context.virtuals (var_ [WEBSITE VIRTUAL].as < vstr_t > ());

    if (var_.count (WMIN WRITE)) context.write_path (var_ [WMIN WRITE].as < ::std::string > ());
    if (var_.count (WMIN STUB)) context.stub (var_ [WMIN STUB].as < ::std::string > ());
    if (var_.count (WMIN HOOK)) context.hook (var_ [WMIN HOOK].as < ::std::string > ());
    if (var_.count (WMIN MACROEND)) context.macro_end (var_ [WMIN MACROEND].as < ::std::string > ());
    if (var_.count (WMIN MACROSTART)) context.macro_start (var_ [WMIN MACROSTART].as < ::std::string > ());
    if (var_.count (WMIN PATH)) context.incoming (path_in_context (var_ [WMIN PATH].as < ::std::string > ()));
    if (var_.count (WMIN MENTION)) context.mentions (var_ [WMIN MENTION].as < vstr_t > ());
    if (var_.count (WMIN TEMPLATE)) context.templates ( var_ [WMIN TEMPLATE].as < vstr_t > ());
    if (var_.count (WMIN TEST_HEADER)) context.test_header (var_ [WMIN TEST_HEADER].as < ::std::string > ());

    context.notify (var_.count (WMOUT NOTIFY));
    context.reset (var_.count (WMOUT RESET));
    if (var_.count (WMOUT SECRET)) context.secret (var_ [WMOUT SECRET].as < ::std::string > ());

    if (context.write_path ().empty ()) context.write_path (context.root ()); }

void pvs (const vstr_t& data)
{   for (auto i : data)
        ::std::cout << i << " "; }

::std::string options::report () const
{   ::std::ostringstream res;
    res << "** Switches:\n";

    if (var_.count (HELP)) res << HELP "\n";
    if (var_.count (VERSION)) res << VERSION "\n";

    if (var_.count (GENERAL FILE)) res << GENERAL FILE ": " << var_ [GENERAL FILE].as< ::std::string > () << "\n";
    if (var_.count (GENERAL NOCHANGE)) res << GENERAL NOCHANGE "\n";
    if (var_.count (GENERAL OUTPUT)) res << GENERAL OUTPUT ": " << var_ [GENERAL OUTPUT].as < int > () << "\n";
    if (var_.count (GENERAL PATH)) res << GENERAL PATH ": " << var_ [GENERAL PATH].as< ::std::string > () << "\n";
    if (var_.count (GENERAL WEBMENTION)) res << GENERAL WEBMENTION "\n";

    if (var_.count (LINK EXTERNAL)) res << LINK EXTERNAL "\n";
    if (var_.count (LINK FORWARD)) res << LINK FORWARD "\n";
    if (var_.count (LINK LINKS)) res << LINK LINKS "\n";
    if (var_.count (LINK ONCE)) res << LINK ONCE "\n";
    if (var_.count (LINK REVOKE)) res << LINK REVOKE "\n";

    if (var_.count (MF EXPORT)) res << MF EXPORT "\n";
    if (var_.count (MF VERIFY)) res << MF VERIFY "\n";

    if (var_.count (SCHEMA MICRODATA)) res << SCHEMA MICRODATA "\n";

    if (var_.count (WEBSITE EXTENSION)) { res << EXTENSION ": "; pvs (var_ [WEBSITE EXTENSION].as < vstr_t > ()); res << "\n"; }
    if (var_.count (WEBSITE INDEX)) res << INDEX ": " << var_ [WEBSITE INDEX].as < ::std::string > () << "\n";
    if (var_.count (WEBSITE SITE)) { res << SITE ": "; pvs (var_ [WEBSITE SITE].as < vstr_t > ()); res << "\n"; }
    if (var_.count (WEBSITE ROOT)) res << ROOT ": " << var_ [WEBSITE ROOT].as < ::std::string > () << "\n";
    if (var_.count (WEBSITE VIRTUAL)) { res << VIRTUAL ": "; pvs (var_ [WEBSITE VIRTUAL].as < vstr_t > ()); res << "\n"; }

    if (var_.count (WMIN WRITE)) { res << WMIN WRITE ": "; var_ [WMIN WRITE].as < ::std::string > (); res << "\n"; }
    if (var_.count (WMIN STUB)) res << WMIN STUB ": " << var_ [WMIN STUB].as< ::std::string > () << "\n";
    if (var_.count (WMIN HOOK)) res << WMIN HOOK ": " << var_ [WMIN HOOK].as< ::std::string > () << "\n";
    if (var_.count (WMIN MACROEND)) res << WMIN MACROEND ": " << var_ [WMIN MACROEND].as< ::std::string > () << "\n";
    if (var_.count (WMIN MACROSTART)) res << WMIN MACROSTART ": " << var_ [WMIN MACROSTART].as< ::std::string > () << "\n";
    if (var_.count (WMIN PATH)) res << WMIN PATH ": " << var_ [WMIN PATH].as< ::std::string > () << "\n";
    if (var_.count (WMIN MENTION)) { res << WMIN MENTION ": "; pvs (var_ [WMIN MENTION].as < vstr_t > ()); res << "\n"; }
    if (var_.count (WMIN TEMPLATE)) { res << WMIN TEMPLATE ": "; pvs (var_ [WMIN TEMPLATE].as < vstr_t > ()); res << "\n"; }
    if (var_.count (WMIN TEST_HEADER)) res << WMIN TEST_HEADER ": " << var_ [WMIN TEST_HEADER].as< ::std::string > () << "\n";

    if (var_.count (WMOUT CLEAR)) res << WMOUT CLEAR "\n";
    if (var_.count (WMOUT NOTIFY)) res << WMOUT NOTIFY "\n";
    if (var_.count (WMOUT RESET)) res << WMOUT RESET "\n";
    if (var_.count (WMOUT SECRET)) res << WMOUT SECRET ": " << var_ [WMOUT SECRET].as< ::std::string > () << "\n";
    res << "\n";
    return res.str (); }
