/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once

#include "common.h"

template < typename VALUE > struct symbol_entry
{   const char* sz_;
    VALUE       v_; };

template < typename VALUE > class symbol_table
{   typedef ssc_map < ::std::string, VALUE > symbol_t;
    typedef ssc_map < VALUE, ::std::string > reverse_t;
    symbol_t symbol_;
    reverse_t reverse_;
    bool wildcards_ = false;
    ::std::string list_;
    void add_to_list (const char* s)
    {   append (list_, ", ", s); }
public:
    void swap (symbol_table& s) noexcept
    {   symbol_.swap (s.symbol_);
        reverse_.swap (s.reverse_);
        ::std::swap (wildcards_, s.wildcards_);
        list_.swap (s.list_); }
    void init (const symbol_entry < VALUE > table [], const size_t size, const bool wildcards = false);
    void extend (const ::std::string& symbol, const VALUE value);
    bool find (const ::std::string& x, VALUE& res);
    bool parse (const ::std::string& x, VALUE& res)
    {   return find (::boost::algorithm::to_lower_copy (trim_the_lot_off (x)), res); }
    ::std::string value_list () const { return list_; }
    ::std::size_t value_count () const { return list_.size (); }
    ::std::string name (const VALUE x)
    { auto i = reverse_.find (x); if (i == reverse_.end ()) return ::std::string (); return i -> second; } };

template < typename VALUE > inline void symbol_table < VALUE > :: init (const symbol_entry < VALUE > table [], const size_t size, const bool wildcards)
{   wildcards_ = wildcards;
    for (int i = 0; i < size; ++i)
    {   if (symbol_.find (table [i].sz_) != symbol_.end ()) ::std::cerr << "ERROR: " << table [i].sz_ << " already in symbol table (new value " << table [i].v_ << ")\n";  // should be assert
        else symbol_.insert (typename symbol_t::value_type (table [i].sz_, table [i].v_));
        if (reverse_.find (table [i].v_) == reverse_.end ()) reverse_.insert (typename reverse_t::value_type (table [i].v_, table [i].sz_));
        add_to_list (table [i].sz_); } }

template < typename VALUE > inline bool symbol_table < VALUE > :: find (const ::std::string& x, VALUE& res)
{   auto i = symbol_.find (x);
    if (i != symbol_.end ())
    {   res = i -> second; return true; }
    if (! wildcards_) return false;
    auto p = x.find ('-');
    if ((p == 0) || (p == x.npos))
    {   p = x.find (':');
        if ((p == 0) || (p == x.npos)) return false; }
    return find (x.substr (0, p), res); }

template < typename VALUE > inline void symbol_table < VALUE > :: extend (const ::std::string& symbol, const VALUE value)
{   symbol_.insert (typename symbol_t::value_type (symbol, value));
    reverse_.insert (typename reverse_t::value_type (value, symbol));
    add_to_list (symbol.c_str ()); }
