/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once

#include "common.h"
#include "url_protocol.h"
#include "url_parameters.h"

class url
{   bool valid_;  // indicates url appears to be correctly formatted, NOT that the target exists
    e_protocol current_;
    protocol protocol_;
    parameters params_;
    void parse (const ::std::string& url, const e_protocol current);
    void clear ()
    {   url e;
        swap (e); }
    void set (const ::std::string& u, const e_protocol current)
    {   parse (u, current); }
public:
    url () : valid_ (true), current_ (pr_http) { }
    url (const ::std::string& u, const e_protocol current = pr_http)
        :   valid_ (true), current_ (current)
    {   set (u, current); }
    url (const url&) = default;
    url (url&&) = default;
    url& operator = (const url&) = default;
    url& operator = (url&&) = default;
    url& operator = (const ::std::string& u)
    {   reset (u);
        return *this; }
    void reset (const url& u)
    {   url f (u);
        swap (f); }
    void reset (const ::std::string& u) // note current protocol carries over as default
    {   e_protocol current = current_;
        clear ();
        set (u, current); }
    void reset ()
    {   clear (); }
    void swap (url& u) noexcept;
    bool empty () const { return ! has_component (es_original); }
    int compare (const url& u) const { return get ().compare (u.get ()); }
    bool has_component (const e_component c) const { return ! get_component (c).empty (); }
    ::std::string get_component (const e_component c) const { return protocol_.get_component (c); }
    bool is_protocol (const e_protocol p) const { return (get_protocol () == p); }
    e_protocol get_protocol () const { return protocol_.get_protocol (); }
    bool has_args () const { return ! params_.empty (); }
    bool has_id () const { return has_component (es_fragment); }
    bool has_path () const { return has_component (es_path); }
    bool has_protocol () const { return ! protocol_.defaulted (); }
    bool is_http () const { return is_protocol (pr_http); }
    bool is_https () const { return is_protocol (pr_https); }
    bool is_local () const { return protocol_.defaulted () && ! empty (); }
    bool is_usable () const { return is_protocol (pr_http) || is_protocol (pr_https); }
    bool invalid () const { return ! valid_; }
    bool tismoi (const url& u) const
    {  return (valid_ && u.valid_ && (protocol_ == u.protocol_)); }
    size_t arg_count () const { return params_.size (); }
    bool exists (const ::std::string& key) const
    { return params_.exists (key); }
    ::std::string value (const ::std::string& key) const
    { return params_.value (key); }
    ::std::string domain () const { return get_component (es_server); }
    ::std::string id () const { return get_component (es_fragment); }
    ::std::string diagnosis () const { return get_component (es_diagnosis); }
    ::std::string original () const { return protocol_.original (); }
    ::std::string get () const
    {   if (invalid ()) return ::std::string (); return protocol_.get (); }
    ::std::string absolute (bool can_use_index = false) const
    {   if (invalid ()) return ::std::string (); return protocol_.absolute (can_use_index); }
    ::std::string page () const { return get_component (es_authority); }
    ::std::string path () const { return get_component (es_path); }
    ::std::string password () const { return get_component (es_password); } };

typedef ::std::vector < url > vurl_t;
vurl_t split_urls_by_space (const ::std::string& s);
