/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

// RFC 3986

#include "url_protocol.h"
#include "url_schemes.h"

const int min_prop_length = 3;
const int max_prop_length = 69;

struct symbol_entry < e_protocol > protocol_symbol_table [] =
{   { PR_FILE, pr_file },
    { PR_FTP, pr_ftp },
    { PR_FTPS, pr_ftps },
    { PR_GEO, pr_geo },
    { PR_HTTP, pr_http },
    { PR_HTTPS, pr_https },
    { PR_MAILTO, pr_mailto },
    { PR_SFTP, pr_sftp } };

struct protocol_server
{   e_protocol protocol_;
    e_scheme scheme_; };

protocol_server ps [] =
{   {   pr_other,   pt_rfc3986 },
    {   pr_file,    pt_local },
    {   pr_ftp,     pt_rfc3986  },
    {   pr_ftps,    pt_rfc3986  },
    {   pr_geo,     pt_geo   },
    {   pr_http,    pt_rfc3986  },
    {   pr_https,   pt_rfc3986  },
    {   pr_ldap,    pt_rfc3986  },
    {   pr_mailto,  pt_rfc3986 },
    {   pr_news,    pt_news  },
    {   pr_sftp,    pt_rfc3986  },
    {   pr_ssh,     pt_rfc3986  },
    {   pr_tel,     pt_tel   },
    {   pr_telnet,  pt_rfc3986  },
    {   pr_urn,     pt_urn   } };

void protocol::init ()
{   symbol::init (protocol_symbol_table, sizeof (protocol_symbol_table) / sizeof (struct symbol_entry < e_protocol >)); }

e_scheme protocol::scheme () const
{   e_protocol prot = symbol < e_protocol > :: get ();
    assert (ps [prot].protocol_ == prot);
    return ps [prot].scheme_; }

bool protocol::parse (const ::std::string& x, const e_protocol current)
{   ::std::string lc (::boost::algorithm::to_lower_copy (trim_the_lot_off (x)));
    if (lc.empty ()) set (current);
    else
    {   ::std::string::size_type colon = lc.find (COLON);
        default_ = (colon == ::std::string::npos);
        if (default_) set (current);
        else if (colon == 0 || colon == lc.length () - 1)
        {   component_ [es_diagnosis] = "invalid protocol and/or missing address";
            return false; }
        else
        {   e_protocol prot;
            if (! symbol < e_protocol > :: parse (lc.substr (0, colon), prot)) set (pr_other);
            else set (prot); }
        if (! url_schemes < SCHEMES > :: parse (scheme (), symbol < e_protocol > :: get (), x, component_))
            return false; }
    return true; }

bool protocol::is_valid () const
{   return url_schemes < SCHEMES > :: is_valid (scheme (), component_, default_); }

::std::string protocol::get () const
{   return url_schemes < SCHEMES > :: get (scheme (), component_, default_); }

::std::string protocol::absolute (bool can_use_index) const
{   return url_schemes < SCHEMES > :: absolute (scheme (), component_, can_use_index, default_); }

bool protocol::operator == (const protocol& rhs) const
{   if (scheme () != rhs.scheme ()) return false;
    return url_schemes < SCHEMES > :: similar (scheme (), component_, rhs.component_); }
