/*
ssc (static site checker)
Copyright (c) 2020 Dylan Harris
https://dylanharris.org/

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public Licence as published by
the Free Software Foundation, either version 3 of the Licence,  or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public Licence for more details.

You should have received a copy of the GNU General Public
Licence along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#pragma once

#include "common.h"
#include "symbol.h"

typedef enum
{   pr_other, pr_file, pr_ftp, pr_ftps, pr_geo, pr_http, pr_https, pr_ldap, pr_mailto, pr_news, pr_sftp, pr_ssh, pr_telnet, pr_tel, pr_urn }
e_protocol;

#define SCHEMES pt_rfc3986, pt_geo, pt_local, pt_news, pt_tel, pt_urn
typedef enum { SCHEMES } e_scheme;

typedef enum { es_diagnosis, es_original, es_query, es_fragment, es_scheme, es_authority, es_user, es_password, es_server, es_port, es_path, es_file } e_component;
constexpr ::std::size_t last_component = static_cast <::std::size_t> (es_file);
constexpr ::std::size_t component_count = last_component + 1;
typedef vstr_t vc_t;

#define PR_FILE "file"
#define PR_FTP "ftp"
#define PR_FTPS "ftps"
#define PR_GEO "geo"
#define PR_HTTP "http"
#define PR_HTTPS "https"
#define PR_MAILTO "mailto"
#define PR_SFTP "sftp"

class protocol : public symbol < e_protocol >
{   bool default_ = false;
    vc_t component_;
public:
    protocol () : default_ (false)
    {   component_.resize (component_count); }
    protocol (const protocol&) = default;
    protocol (protocol&&) = default;
    protocol (const ::std::string& x, const e_protocol current = pr_https)
    {   component_.resize (component_count);
        parse (x, current); }
    protocol& operator = (const protocol&) = default;
    protocol& operator = (protocol&&) = default;
    ~protocol () = default;
    bool operator == (const protocol& rhs) const;
    bool parse (const ::std::string& x, e_protocol current = pr_https);
    void swap (protocol& p) noexcept
    {   ::std::swap (default_, p.default_);
        component_.swap (p.component_);
        symbol < e_protocol > :: swap (p); }
    bool is_valid () const;
    ::std::string get () const;
    ::std::string absolute (bool can_use_index) const;
    bool defaulted () const { return default_; }
    bool unknown () const { return get_protocol () == pr_other; }
    bool has_component (const e_component c) const { return ! get_component (c).empty (); }
    ::std::string original () const { return component_ [es_original]; }
    ::std::string get_component (const e_component c) const { return component_ [c]; }
    void set_component (const e_component c, const ::std::string& val) { component_ [c] = val; }
    e_protocol get_protocol () const { return (symbol < e_protocol > :: get ()); }
    bool is_protocol (const e_protocol p) const { return (get_protocol () == p); }
    e_scheme scheme () const;
    static void init (); };
